// Speedometer GPX Loader – v3.0 (User‑Verified Max & Enhanced Smoothing)
// ------------------------------------------------------------------------
// Adds a dialog so the user can confirm or correct the maximum speed (mph)
// detected in the GPX track and applies a moving‑average filter to remove
// GPS spike artefacts before proceeding with the original workflow.

(function () {
    app.beginUndoGroup("Speedometer GPX Upload v3.0");

    /* ========= 1. Duplicate template comp ========= */
    var templateName = "Speedometer";
    var templateComp = null;
    for (var i = 1; i <= app.project.numItems; i++) {
        var it = app.project.item(i);
        if (it instanceof CompItem && it.name === templateName) { templateComp = it; break; }
    }
    if (!templateComp) { alert("Template comp ‘"+templateName+"’ not found."); return; }

    // Increment name: Speedometer, Speedometer 2, 3, …
    var maxNum = 0, reNum = /^Speedometer(?: (\d+))?$/;
    for (i = 1; i <= app.project.numItems; i++) {
        var m = reNum.exec(app.project.item(i).name);
        if (m && m[1]) maxNum = Math.max(maxNum, parseInt(m[1],10));
    }
    var comp = templateComp.duplicate();
    comp.name = maxNum ? ("Speedometer " + (maxNum+1)) : "Speedometer 2";

    /* ========= 2. Grab bar layers ========= */
    var speedLayer = comp.layer("SpeedBar");
    var topLayer   = comp.layer("TopSpeedBar");
    if (!speedLayer || !topLayer) { alert("‘SpeedBar’ / ‘TopSpeedBar’ layers missing."); return; }
    var sP = speedLayer.property("Scale"), tP = topLayer.property("Scale");

    /* ========= 3. Select GPX ========= */
    var file = File.openDialog("Select GPX file", "GPX:*.gpx");
    if (!file || !file.open("r")) { alert("Import canceled."); return; }
    var xml = file.read(); file.close();

    /* ========= 4. Parse track‑points ========= */
    var RE = /<trkpt[^>]*lat="([^"]+)"[^>]*lon="([^"]+)"[^>]*>[\s\S]*?<time>([^<]+)<\/time>/g;
    var pts = [], m;
    while (m = RE.exec(xml)) {
        var d = m[3].match(/^(\d{4})-(\d{2})-(\d{2})T(\d{2}):(\d{2}):(\d{2})/);
        if (!d) continue;
        var t = Date.UTC(+d[1], +d[2]-1, +d[3], +d[4], +d[5], +d[6]) / 1000;
        pts.push({lat:+m[1], lon:+m[2], time:t});
    }
    if (pts.length < 2) { alert("GPX file has too few points."); return; }

    /* ========= 5. Compute ‘instantaneous’ speed (m/s) ========= */
    function rad(x){ return x*Math.PI/180; }
    function hav(a,b,c,d){ var R=6371000, dLat=rad(c-a), dLon=rad(d-b);
        var f = Math.sin(dLat/2), g = Math.sin(dLon/2);
        var A = f*f + Math.cos(rad(a))*Math.cos(rad(c))*g*g;
        return R*2*Math.atan2(Math.sqrt(A), Math.sqrt(1-A));
    }
    var times=[0], speeds=[0], t0=pts[0].time;
    for (i=1; i<pts.length; i++){
        var p0=pts[i-1], p1=pts[i], dt=p1.time-p0.time;
        if (dt<=0) continue;
        times.push(p1.time - t0);
        speeds.push( hav(p0.lat,p0.lon, p1.lat,p1.lon) / dt );
    }
    comp.duration = times[times.length-1] + 1;

    /* ========= 5b. Verify / correct max speed ========= */
    var mphSpeeds = [], rawMaxMPH = 0;
    for (i = 0; i < speeds.length; i++) {
        var mphVal = speeds[i] * 2.23694;
        mphSpeeds.push(mphVal);
        if (mphVal > rawMaxMPH) rawMaxMPH = mphVal;
    }

    // Ask the user to confirm the max speed
    var dlg = new Window("dialog", "Confirm Max Speed");
    dlg.orientation = "column";
    dlg.alignChildren = ["left", "top"];
    dlg.add("statictext", undefined, "Detected max speed (mph):");
    var txt = dlg.add("edittext", undefined, rawMaxMPH.toFixed(1));
    txt.characters = 8;
    txt.active = true;
    dlg.add("statictext", undefined, "Edit if incorrect, then click OK.");
    var btnGrp = dlg.add("group");
    btnGrp.orientation = "row";
    var okBtn = btnGrp.add("button", undefined, "OK", {name:"ok"});
    var cancelBtn = btnGrp.add("button", undefined, "Cancel", {name:"cancel"});
    if (dlg.show() != 1) { alert("Operation canceled."); return; }
    var userMaxMPH = parseFloat(txt.text);
    if (isNaN(userMaxMPH) || userMaxMPH <= 0) userMaxMPH = rawMaxMPH;
    var userMaxMS = userMaxMPH / 2.23694;

    /* ========= 5c. Smooth & clamp speeds ========= */
    function movingAverage(arr, radius) {
        var out = [];
        for (var n = 0; n < arr.length; n++) {
            var sum = 0, cnt = 0;
            for (var k = -radius; k <= radius; k++) {
                var idx2 = n + k;
                if (idx2 < 0 || idx2 >= arr.length) continue;
                sum += arr[idx2];
                cnt++;
            }
            out.push(sum / cnt);
        }
        return out;
    }
    var smoothed = movingAverage(speeds, 2); // 5‑sample window (2 left + self + 2 right)
    for (i = 0; i < smoothed.length; i++) {
        if (smoothed[i] > userMaxMS) smoothed[i] = userMaxMS;
        if (smoothed[i] < 0) smoothed[i] = 0;
    }
    speeds = smoothed.slice();

    /* ========= 6. Post‑process speeds ========= */
    var LOW_THRESH = 5 / 2.23694;  // mph → m/s
    var CRASH_DROP = 10;          // mph

    var crashed = false;
    for (i=1; i<speeds.length; i++){
        var drop = (speeds[i-1] - speeds[i]) * 2.23694;
        if (!crashed && drop >= CRASH_DROP) crashed = true;

        if (crashed) {
            if (speeds[i]*2.23694 < LOW_THRESH * 2.23694) speeds[i] = 0;
            else crashed = false;
        }
        if (speeds[i] < LOW_THRESH) speeds[i] = 0;
    }

    /* ========= 6b. Smooth false spikes (accel check) ========= */
    var MAX_ACCEL = 5; // m/s² threshold
    for (i=1; i<speeds.length-1; i++){
        var dt = times[i] - times[i-1];
        if (dt <= 0) continue;
        var accel = (speeds[i] - speeds[i-1]) / dt;
        if (Math.abs(accel) > MAX_ACCEL) {
            // interpolate between neighbors
            speeds[i] = (speeds[i-1] + speeds[i+1]) / 2;
        }
    }

    /* ========= 7. Generate keyframes & markers ========= */
    var maxSeen = 0, mph;
    sP.setValueAtTime(0,[100,0]); tP.setValueAtTime(0,[100,0]);

    var milestones = [20,25,30,35];
    var milestoneHit = {}, rawPeak = 0, rawPeakIdx = 0;

    for (i=0; i<times.length; i++){
        mph = speeds[i] * 2.23694;
        sP.setValueAtTime(times[i],[100,mph]);
        maxSeen = Math.max(maxSeen, mph);
        tP.setValueAtTime(times[i],[100,maxSeen]);
        if (mph > rawPeak){ rawPeak = mph; rawPeakIdx = i; }
        for (var mi=0; mi<milestones.length; mi++){
            var mVal = milestones[mi];
            if (mph >= mVal && milestoneHit[mVal] === undefined){
                milestoneHit[mVal] = times[i];
                comp.markerProperty.setValueAtTime(times[i], new MarkerValue(mVal+"mph"));
            }
        }
    }
    comp.markerProperty.setValueAtTime(times[rawPeakIdx], new MarkerValue("Max " + rawPeak.toFixed(1) + " mph"));

    app.endUndoGroup();
})();
